<?php

namespace App\Http\Controllers;

use App\Models\Blog;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Services\BunnyCDNService;
use Illuminate\Support\Facades\Validator;

class BlogController extends Controller
{

    protected $bunnyCDNService;

    public function __construct(BunnyCDNService $bunnyCDNService)
    {
        $this->bunnyCDNService = $bunnyCDNService;
    }



    public function create(Request $request)
    {


        $validator = Validator::make($request->all(), [
            'slug' => 'nullable|string',
            'title' => 'nullable|string',
            'meta_title' => 'nullable|string',
            'meta_description' => 'nullable|string',
            'content' => 'nullable|string',
            'description' => 'nullable|string',
            'image' => 'nullable|string',
            'category_id' => 'required|exists:categories,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        $uploadedImageUrl = null;
        $base64Image = $request->input('image');

        if (!empty($base64Image)) {
            $fileName = Str::uuid() . '.jpg';
            $imageData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $base64Image));

            $tempDirectory = storage_path('app/public/temp');
            $tempFilePath = $tempDirectory . '/' . $fileName;

            if (!file_exists($tempDirectory)) {
                mkdir($tempDirectory, 0777, true);
            }

            file_put_contents($tempFilePath, $imageData);

            $cdnUrl = $this->bunnyCDNService->uploadImage($tempFilePath, $fileName);

            if ($cdnUrl) {
                $uploadedImageUrl = $cdnUrl;
            }

            unlink($tempFilePath);
        }


        Blog::create([
            'image' => $uploadedImageUrl,
            'title' => $request->title,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
            'content' => $request->content,
            'description' => $request->description,
            'slug' => $request->slug,
            'category_id' => $request->category_id,

        ]);

        return response()->json(['message' => 'Blog updated successfully.'], Response::HTTP_OK);
    }



    public function update(Request $request, $id)
    {
        // Fetch the blog
        $blog = Blog::find($id);

        if (!$blog) {
            return response()->json([
                'message' => 'Blog not found or you don\'t have permission to edit this blog.'
            ], Response::HTTP_NOT_FOUND);
        }

        // Validate the request
        $validator = Validator::make($request->all(), [
            'slug' => 'nullable|string',
            'title' => 'nullable|string',
            'meta_title' => 'nullable|string',
            'meta_description' => 'nullable|string',
            'content' => 'nullable|string',
            'description' => 'nullable|string',
            'image' => 'nullable|string',
            'category_id' => 'required|exists:categories,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        // Process the image if provided
        $uploadedImageUrl = $blog->image; // Retain the current image URL
        $base64Image = $request->input('image');

        if (!empty($base64Image) && $base64Image !== $blog->image) {
            try {
                $fileName = Str::uuid() . '.jpg';
                $imageData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $base64Image));

                if ($imageData === false) {
                    return response()->json([
                        'message' => 'Invalid base64 image data.'
                    ], Response::HTTP_BAD_REQUEST);
                }

                $tempDirectory = storage_path('app/public/temp');
                $tempFilePath = $tempDirectory . '/' . $fileName;

                if (!file_exists($tempDirectory)) {
                    mkdir($tempDirectory, 0777, true);
                }

                file_put_contents($tempFilePath, $imageData);

                $cdnUrl = $this->bunnyCDNService->uploadImage($tempFilePath, $fileName);

                if ($cdnUrl) {
                    $uploadedImageUrl = $cdnUrl;
                }

                unlink($tempFilePath);
            } catch (\Exception $e) {
                return response()->json([
                    'message' => 'Failed to process the image.',
                    'error' => $e->getMessage(),
                ], Response::HTTP_INTERNAL_SERVER_ERROR);
            }
        }

        // Update the blog
        $blog->update([
            'slug' => $request->input('slug', $blog->slug),
            'title' => $request->input('title', $blog->title),
            'meta_title' => $request->input('meta_title', $blog->meta_title),
            'meta_description' => $request->input('meta_description', $blog->meta_description),
            'content' => $request->input('content', $blog->content),
            'description' => $request->input('description', $blog->description),
            'image' => $uploadedImageUrl,
            'category_id' => $request->input('category_id'),
        ]);

        return response()->json([
            'message' => 'Blog updated successfully.',
            'data' => $blog
        ], Response::HTTP_OK);
    }

    public function search(Request $request)
    {
        try {
            $searchValue = $request->input('searchValue');
            $regex = strtolower($searchValue);

            $page = $request->input('page', 1);
            $perPage = $request->input('per_page', 10);

            $query = Blog::whereNull('deleted_at')
                ->with('category');

            if ($searchValue) {
                $query->where(function ($query) use ($regex) {
                    $query->whereRaw('LOWER(title) LIKE ?', ["%{$regex}%"]);
                });
            }

            // Paginate the results
            $blogs = $query->paginate($perPage, ['*'], 'page', $page);

            return response()->json([
                'message' => 'Blogs retrieved successfully.',
                'data' => $blogs->items(),
                'total' => $blogs->total(),
                'current_page' => $blogs->currentPage(),
                'last_page' => $blogs->lastPage(),
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Internal Server Error',
                'error' => $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }


    public function get($id)
    {


        $query = Blog::where('id', $id)
            ->whereNull('deleted_at');

        $blog = $query->first();

        if (!$blog) {
            return response()->json(['message' => 'No blog found.'], Response::HTTP_NOT_FOUND);
        }

        return response()->json([
            'message' => 'Blog retrieved successfully.',
            'data' => $blog
        ], Response::HTTP_OK);
    }





    public function destroy($id, Request $request)
    {

        $blog = Blog::find($id);

        if (!$blog) {
            return response()->json(['message' => 'Blog not found or you don\'t have permission to delete this blog.'], Response::HTTP_NOT_FOUND);
        }

        $blog->deleted_at = now();
        $blog->save();

        return response()->json(['message' => 'Blog deleted successfully.', 'data' => $blog], Response::HTTP_OK);
    }



    public function searchClient(Request $request)
    {
        try {
            $page = $request->input('page', 1);
            $perPage = $request->input('per_page', 10);
            $category = $request->input('category');
            $sort = $request->input('sort', '');


            $query = Blog::query()
                ->whereNull('deleted_at')
                ->with('category');

            if (!empty($category)) {
                $query->where('category_id', $category);
            }


            if (!empty($sort)) {
                if ($sort === 'newest') {
                    $query->orderBy('created_at', 'desc');
                } elseif ($sort === 'oldest') {
                    $query->orderBy('created_at', 'asc');
                } elseif ($sort === 'a-z') {
                    $query->orderBy('name', 'asc');
                } elseif ($sort === 'z-a') {
                    $query->orderBy('name', 'desc');
                }
            } else {
                $query->orderBy('created_at', 'desc');
            }

            $blogs = $query->paginate($perPage, ['*'], 'page', $page);

            return response()->json([
                'message' => 'Blogs retrieved successfully.',
                'data' => $blogs->items(),
                'total' => $blogs->total(),
                'current_page' => $blogs->currentPage(),
                'last_page' => $blogs->lastPage(),
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            // Return a JSON response for any server error
            return response()->json([
                'message' => 'An error occurred while retrieving blogs.',
                'error' => $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }


    public function getClient($id)
    {


        $query = Blog::where('id', $id)
            ->whereNull('deleted_at')
            ->with('category');

        $blog = $query->first();

        if (!$blog) {
            return response()->json(['message' => 'No blog found.'], Response::HTTP_NOT_FOUND);
        }

        return response()->json([
            'message' => 'Blog retrieved successfully.',
            'data' => $blog
        ], Response::HTTP_OK);
    }
}
