<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Http\Response;
use App\Services\BunnyCDNService;
use Illuminate\Support\Facades\Validator;

class BrandController extends Controller
{
    protected $bunnyCDNService;

    public function __construct(BunnyCDNService $bunnyCDNService)
    {
        $this->bunnyCDNService = $bunnyCDNService;
    }
    public function createBrand(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100',
            'description' => 'nullable|string',
            'slug' => 'required|string',
            'image' => 'nullable|string',
            'featured' => 'nullable|boolean',

        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        $uploadedImageUrl = null;
        if ($base64Image = $request->input('image')) {
            $fileName = Str::uuid() . '.jpg';
            $imageData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $base64Image));

            $tempDirectory = storage_path('app/public/temp');
            $tempFilePath = $tempDirectory . '/' . $fileName;

            if (!file_exists($tempDirectory)) {
                mkdir($tempDirectory, 0777, true);
            }

            file_put_contents($tempFilePath, $imageData);

            $uploadedImageUrl = $this->bunnyCDNService->uploadImage($tempFilePath, $fileName);

            unlink($tempFilePath);
        }


        $brand = Brand::create([
            'name' => $request->name,
            'slug' => $request->slug,
            'description' => $request->description,
            'image' => $uploadedImageUrl,
            'featured' => $request->featured,

        ]);

        return response()->json([
            'message' => 'Brand created successfully.',
        ], Response::HTTP_CREATED);
    }
    public function updateBrand(Request $request, $id)
    {
        $brand = Brand::find($id);

        if (!$brand) {
            return response()->json(['message' => 'Brand not found.'], Response::HTTP_NOT_FOUND);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100',
            'description' => 'nullable|string',
            'slug' => 'required|string',
            'image' => 'nullable|string',
            'featured' => 'nullable|boolean',

        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        $uploadedImageUrl = $brand->image;
        $base64Image = $request->input('image');

        if (!empty($base64Image) && $base64Image !== $brand->image) {
            $fileName = Str::uuid() . '.jpg';
            $imageData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $base64Image));

            $tempDirectory = storage_path('app/public/temp');
            $tempFilePath = $tempDirectory . '/' . $fileName;

            if (!file_exists($tempDirectory)) {
                mkdir($tempDirectory, 0777, true);
            }

            file_put_contents($tempFilePath, $imageData);

            $cdnUrl = $this->bunnyCDNService->uploadImage($tempFilePath, $fileName);

            if ($cdnUrl) {
                $uploadedImageUrl = $cdnUrl;
            }

            unlink($tempFilePath);
        }
        $brand->image = $uploadedImageUrl;
        $brand->slug = $request->slug;
        $brand->featured = $request->featured;

        $brand->name = $request->name;
        $brand->description = $request->description;
        $brand->save();

        return response()->json(['message' => 'Brand updated successfully.', 'data' => $brand], Response::HTTP_OK);
    }


    public function delete($id)
    {
        $brand = Brand::find($id);

        if (!$brand) {
            return response()->json(['message' => 'Brand not found.'], Response::HTTP_NOT_FOUND);
        }


        $brand->delete();

        return response()->json(['message' => 'Brand deleted successfully.'], Response::HTTP_OK);
    }


    public function search(Request $request)
    {
        try {
            $searchValue = $request->input('searchValue');
            $regex = strtolower($searchValue);

            $page = $request->input('page');
            $perPage = $request->input('per_page' ,10);

            $query = Brand::whereNull('deleted_at')
                ->orderBy('created_at', 'desc');

            if ($searchValue) {
                $query->where(function ($query) use ($regex) {
                    $query->whereRaw('LOWER(name) LIKE ?', ["%{$regex}%"]);
                });
            }

            if ($page) {


                $brands = $query->paginate($perPage, ['*'], 'page', $page);

                return response()->json([
                    'message' => 'Brands retrieved successfully.',
                    'data' => $brands->items(),
                    'total' => $brands->total(),
                    'current_page' => $brands->currentPage(),
                    'last_page' => $brands->lastPage(),
                ], Response::HTTP_OK);
            } else {
                $brands = $query->get();

                return response()->json([
                    'message' => 'Brands retrieved successfully.',
                    'data' => $brands,

                ], Response::HTTP_OK);
            }
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Internal Server Error',
                'error' => $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }




    public function generateUniqueSlug($name, $model)
    {
        $slug = Str::slug($name);

        $originalSlug = $slug;
        $count = 1;

        while ($model::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $count;
            $count++;
        }

        return $slug;
    }

    public function importBrands(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xls,xlsx|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 400);
        }

        try {
            $data = Excel::toCollection(null, $request->file('file'))->first();

            $importedCount = 0;
            $skippedCount = 0;

            foreach ($data as $key => $row) {
                if ($key === 0) {
                    continue;
                }

                if (empty($row[1])) {
                    $skippedCount++;
                    continue;
                }

                $slug = $this->generateUniqueSlug($row[1], Brand::class);

                $description = $row[3] ?? 'No description provided';

                Brand::create([
                    'name' => $row[1],
                    'slug' => $slug,
                    'description' => $description,
                ]);

                $importedCount++;
            }

            return response()->json([
                'success' => true,
                'message' => 'Brands imported successfully!',
                'imported' => $importedCount,
                'skipped' => $skippedCount,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'There was an error processing the file: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function searchClient(Request $request)
    {
        try {
            $featured = $request->input('featured');

            $query = Brand::query()
                ->whereNull('deleted_at')
                ->orderBy('created_at', 'desc');

            if (!is_null($featured)) {
                $query->where('featured', filter_var($featured, FILTER_VALIDATE_BOOLEAN));
            }

            $brands = $query->get();

            return response()->json([
                'message' => 'Brands retrieved successfully.',
                'data' => $brands,
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            // Return a JSON response for any server error
            return response()->json([
                'message' => 'An error occurred while retrieving categories.',
                'error' => $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}
