<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderProduct;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class StatisticsController extends Controller
{
    public function dashboard(Request $request)
    {
        $tz           = config('app.timezone', 'Africa/Casablanca');
        $to           = $request->date('to')   ?: Carbon::now($tz)->endOfDay();
        $from         = $request->date('from') ?: (clone $to)->copy()->subMonths(6)->startOfDay();
        $monthsCount  = max(1, (int) $request->get('months', 7));
        $topProductsN = max(1, (int) $request->get('top_products', 5));
        $productId    = $request->integer('product_id'); // ← NEW

        $defaultStatuses = ['paid', 'processing', 'shipped', 'completed'];
        $statuses        = $request->has('statuses')
            ? array_filter((array) $request->get('statuses'))
            : $defaultStatuses;

        $cacheKey = sprintf(
            'stats:dashboard:%s:%s:%s:%d:%d:%s',
            optional($from)->format('Y-m-d'),
            optional($to)->format('Y-m-d'),
            implode(',', $statuses),
            $monthsCount,
            $topProductsN,
            $productId ?? 'all'
        );

        return Cache::remember($cacheKey, now()->addMinutes(5), function () use ($from, $to, $statuses, $monthsCount, $topProductsN, $productId) {

            /* ------------ base orders in window ------------ */
            $ordersQ = Order::query()
                ->whereBetween('created_at', [$from, $to])
                ->when(!empty($statuses), fn($q) => $q->whereIn('status', $statuses));

            $orderIds = (clone $ordersQ)->pluck('id');

            /* ------------ KPIs: revenue, orders, AOV ------------ */
            $revenue = OrderProduct::query()
                ->whereIn('order_id', $orderIds)
                ->when($productId, fn($q) => $q->where('product_id', $productId))   // ← product filter
                ->select(DB::raw('COALESCE(SUM(quantity * price), 0) as gross'))
                ->value('gross');

            $ordersCount = (clone $ordersQ)
                ->when($productId, function ($q) use ($productId) {
                    // Only count orders that actually include the product
                    $q->whereIn('id', function ($sub) use ($productId) {
                        $sub->select('order_id')
                            ->from('order_products')
                            ->where('product_id', $productId);
                    });
                })
                ->count();

            $aov = $ordersCount > 0 ? ($revenue / $ordersCount) : 0;
            $conversionRate = null;

            /* ------------ Monthly revenue series (last N months) ------------ */
            $monthBuckets = [];
            $cursor = Carbon::parse($to)->startOfMonth()->subMonths($monthsCount - 1);
            for ($i = 0; $i < $monthsCount; $i++) {
                $key = $cursor->format('Y-m');
                $monthBuckets[$key] = [
                    'name'  => $cursor->format('M'),
                    'total' => 0,
                ];
                $cursor->addMonth();
            }

            $monthly = OrderProduct::query()
                ->join('orders', 'order_products.order_id', '=', 'orders.id')
                ->whereBetween('orders.created_at', [
                    Carbon::parse($to)->copy()->startOfMonth()->subMonths($monthsCount - 1),
                    Carbon::parse($to)->copy()->endOfMonth(),
                ])
                ->when(!empty($statuses), fn($q) => $q->whereIn('orders.status', $statuses))
                ->when($productId, fn($q) => $q->where('order_products.product_id', $productId)) // ← product filter
                ->groupBy(DB::raw("DATE_FORMAT(orders.created_at, '%Y-%m')"))
                ->orderBy(DB::raw("DATE_FORMAT(orders.created_at, '%Y-%m')"))
                ->get([
                    DB::raw("DATE_FORMAT(orders.created_at, '%Y-%m') as ym"),
                    DB::raw('COALESCE(SUM(order_products.quantity * order_products.price), 0) as total'),
                ]);

            foreach ($monthly as $row) {
                if (isset($monthBuckets[$row->ym])) {
                    $monthBuckets[$row->ym]['total'] = (float) $row->total;
                }
            }
            $revenueData = array_values($monthBuckets);

            /* ------------ Top products (or single product if filtered) ------------ */
            if ($productId) {
                $one = OrderProduct::query()
                    ->join('orders', 'order_products.order_id', '=', 'orders.id')
                    ->join('products', 'order_products.product_id', '=', 'products.id')
                    ->whereBetween('orders.created_at', [$from, $to])
                    ->when(!empty($statuses), fn($q) => $q->whereIn('orders.status', $statuses))
                    ->where('order_products.product_id', $productId)
                    ->groupBy('order_products.product_id', 'products.name')
                    ->get([
                        'products.name as name',
                        DB::raw('COALESCE(SUM(order_products.quantity),0) as total'),
                    ])
                    ->map(fn($r) => ['name' => $r->name ?? 'Unnamed Product', 'total' => (int) $r->total])
                    ->values()
                    ->all();

                $productData = $one; // single item (or empty if no sales in range)
            } else {
                $productData = OrderProduct::query()
                    ->join('orders', 'order_products.order_id', '=', 'orders.id')
                    ->join('products', 'order_products.product_id', '=', 'products.id')
                    ->whereBetween('orders.created_at', [$from, $to])
                    ->when(!empty($statuses), fn($q) => $q->whereIn('orders.status', $statuses))
                    ->groupBy('order_products.product_id', 'products.name')
                    ->orderByDesc(DB::raw('SUM(order_products.quantity)'))
                    ->limit($topProductsN)
                    ->get([
                        'products.name as name',
                        DB::raw('SUM(order_products.quantity) as total'),
                    ])
                    ->map(fn($r) => ['name' => $r->name ?? 'Unnamed Product', 'total' => (int) $r->total])
                    ->values()
                    ->all();
            }

            /* ------------ Deltas vs previous window ------------ */
            $prevFrom = (clone $from)->copy()->subMonths(1);
            $prevTo   = (clone $to)->copy()->subMonths(1);

            $prevOrdersQ = Order::query()
                ->whereBetween('created_at', [$prevFrom, $prevTo])
                ->when(!empty($statuses), fn($q) => $q->whereIn('status', $statuses));

            $prevOrderIds = (clone $prevOrdersQ)
                ->when($productId, function ($q) use ($productId) {
                    $q->whereIn('id', function ($sub) use ($productId) {
                        $sub->select('order_id')->from('order_products')->where('product_id', $productId);
                    });
                })
                ->pluck('id');

            $prevRevenue = OrderProduct::query()
                ->whereIn('order_id', $prevOrderIds)
                ->when($productId, fn($q) => $q->where('product_id', $productId))
                ->select(DB::raw('COALESCE(SUM(quantity * price), 0) as gross'))
                ->value('gross');

            $prevOrdersCount = (clone $prevOrdersQ)
                ->when($productId, function ($q) use ($productId) {
                    $q->whereIn('id', function ($sub) use ($productId) {
                        $sub->select('order_id')->from('order_products')->where('product_id', $productId);
                    });
                })
                ->count();

            $prevAov = $prevOrdersCount > 0 ? ($prevRevenue / $prevOrdersCount) : 0;

            $delta = function ($current, $previous) {
                if ((float)$previous == 0.0) {
                    return $current > 0 ? 100.0 : 0.0;
                }
                return (($current - $previous) / $previous) * 100.0;
            };

            $kpis = [
                'totalRevenue'      => round((float) $revenue, 2),
                'totalOrders'       => (int) $ordersCount,
                'averageOrderValue' => round((float) $aov, 2),
                'conversionRate'    => null,
                'deltas' => [
                    'totalRevenue'      => round($delta((float) $revenue, (float) $prevRevenue), 1),
                    'totalOrders'       => round($delta((float) $ordersCount, (float) $prevOrdersCount), 1),
                    'averageOrderValue' => round($delta((float) $aov, (float) $prevAov), 1),
                ],
            ];

            return response()->json([
                'range' => [
                    'from'      => $from->toISOString(),
                    'to'        => $to->toISOString(),
                    'statuses'  => $statuses,
                    'productId' => $productId,
                ],
                'kpis'         => $kpis,
                'revenueData'  => $revenueData,
                'productData'  => $productData,
            ]);
        });
    }
}
